<?php

namespace Modules\Tickets\Services;

use Corbital\LaravelEmails\Facades\Email;
use Modules\Tickets\Models\Ticket;
use Modules\Tickets\Models\TicketReply;

class NotificationService
{
    /**
     * Send ticket created notification
     */
    public function sendTicketCreatedNotification(Ticket $ticket): void
    {
        if (! config('tickets.notifications.enabled', true)) {
            return;
        }

        try {
            // Send to client
            if (config('tickets.notifications.send_to_client.ticket_created', true)) {
                $this->sendClientTicketCreatedEmail($ticket);
            }

            // Send to admin
            if (config('tickets.notifications.send_to_admin.ticket_created', true)) {
                $this->sendAdminTicketCreatedEmail($ticket);
            }
        } catch (\Exception $e) {
            app_log('Failed to send ticket created notifications', 'error', $e, [
                'ticket_id' => $ticket->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Send ticket reply notification
     */
    public function sendTicketReplyNotification(TicketReply $reply): void
    {
        if (! config('tickets.notifications.enabled', true)) {
            return;
        }

        try {
            $ticket = $reply->ticket;

            // If reply is from admin, notify client
            if ($reply->user->hasRole('admin') && config('tickets.notifications.send_to_client.reply_received', true)) {
                $this->sendClientReplyReceivedEmail($ticket, $reply);
            }

            // If reply is from client, notify admin
            if (! $reply->user->hasRole('admin') && config('tickets.notifications.send_to_admin.reply_received', true)) {
                $this->sendAdminReplyReceivedEmail($ticket, $reply);
            }
        } catch (\Exception $e) {
            app_log('Failed to send ticket reply notifications', 'error', $e, [
                'reply_id' => $reply->id,
                'ticket_id' => $reply->ticket_id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Send ticket status changed notification
     */
    public function sendTicketStatusChangedNotification(Ticket $ticket, string $oldStatus): void
    {
        if (! config('tickets.notifications.enabled', true)) {
            return;
        }

        try {
            // Send to client
            if (config('tickets.notifications.send_to_client.ticket_updated', true)) {
                $this->sendClientStatusChangedEmail($ticket, $oldStatus);
            }

            // Send to admin if needed
            if (config('tickets.notifications.send_to_admin.ticket_updated', true)) {
                $this->sendAdminStatusChangedEmail($ticket, $oldStatus);
            }
        } catch (\Exception $e) {
            app_log('Failed to send ticket status changed notifications', 'error', $e, [
                'ticket_id' => $ticket->id,
                'old_status' => $oldStatus,
                'new_status' => $ticket->status,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Send client ticket created email
     */
    protected function sendClientTicketCreatedEmail(Ticket $ticket): void
    {
        $client = $ticket->tenant;

        if (is_smtp_valid()) {
            Email::to($client->email)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - Ticket Created')
                ->template('tickets::emails.client.ticket-created', [
                    'client' => $client,
                    'ticket' => $ticket,
                    'login_url' => tenant_route('tenant.tickets.show', ['ticket' => $ticket->id]),
                ])
                ->send();
        }
    }

    /**
     * Send admin ticket created email
     */
    protected function sendAdminTicketCreatedEmail(Ticket $ticket): void
    {
        $adminEmail = config('tickets.notifications.admin_email') ?? config('mail.from.address');

        if (! $adminEmail) {
            return;
        }

        if (is_smtp_valid()) {
            Email::to($adminEmail)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - New Ticket Created')
                ->template('tickets::emails.admin.ticket-created', [
                    'ticket' => $ticket,
                    'client' => $ticket->client,
                    'admin_url' => route('admin.tickets.show', $ticket->id),
                ])
                ->send();
        }

    }

    /**
     * Send client reply received email
     */
    protected function sendClientReplyReceivedEmail(Ticket $ticket, TicketReply $reply): void
    {
        $client = $ticket->client;

        if (is_smtp_valid()) {
            Email::to($client->email)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - New Reply')
                ->template('tickets::emails.client.reply-received', [
                    'client' => $client,
                    'ticket' => $ticket,
                    'reply' => $reply,
                    'login_url' => tenant_route('tenant.tickets.show', ['ticket' => $ticket->id]),
                ])
                ->send();
        }
    }

    /**
     * Send admin reply received email
     */
    protected function sendAdminReplyReceivedEmail(Ticket $ticket, TicketReply $reply): void
    {
        $adminEmail = config('tickets.notifications.admin_email') ?? config('mail.from.address');

        if (! $adminEmail) {
            return;
        }

        if (is_smtp_valid()) {
            Email::to($adminEmail)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - New Reply from Client')
                ->template('tickets::emails.admin.reply-received', [
                    'ticket' => $ticket,
                    'reply' => $reply,
                    'client' => $ticket->client,
                    'admin_url' => route('admin.tickets.show', $ticket->id),
                ])
                ->send();
        }
    }

    /**
     * Send client status changed email
     */
    protected function sendClientStatusChangedEmail(Ticket $ticket, string $oldStatus): void
    {
        $client = $ticket->client;

        if (is_smtp_valid()) {
            Email::to($client->email)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - Status Updated')
                ->template('tickets::emails.client.status-changed', [
                    'client' => $client,
                    'ticket' => $ticket,
                    'old_status' => $oldStatus,
                    'new_status' => $ticket->status,
                    'login_url' => tenant_route('tenant.tickets.show', ['ticket' => $ticket->id]),
                ])
                ->send();
        }
    }

    /**
     * Send admin status changed email
     */
    protected function sendAdminStatusChangedEmail(Ticket $ticket, string $oldStatus): void
    {
        $adminEmail = config('tickets.notifications.admin_email') ?? config('mail.from.address');

        if (! $adminEmail) {
            return;
        }

        // Only send to admin if status was changed by client
        if (! $ticket->wasRecentlyUpdatedBy('admin') && is_smtp_valid()) {
            Email::to($adminEmail)
                ->subject('[Ticket #'.$ticket->ticket_id.'] - Status Changed by Client')
                ->template('tickets::emails.admin.status-changed', [
                    'ticket' => $ticket,
                    'client' => $ticket->client,
                    'old_status' => $oldStatus,
                    'new_status' => $ticket->status,
                    'admin_url' => route('admin.tickets.show', $ticket->id),
                ])
                ->send();
        }
    }

    /**
     * Send bulk notifications for multiple tickets
     */
    public function sendBulkNotifications(array $tickets, string $action, array $data = []): void
    {
        foreach ($tickets as $ticket) {
            try {
                switch ($action) {
                    case 'status_changed':
                        $this->sendTicketStatusChangedNotification($ticket, $data['old_status'] ?? 'unknown');
                        break;
                    case 'closed':
                        $this->sendTicketClosedNotification($ticket);
                        break;
                    default:
                        app_log('Unknown bulk notification action', 'warning', null, [
                            'ticket_id' => $ticket->id,
                            'action' => $action,
                        ]);
                        break;
                }
            } catch (\Exception $e) {
                app_log('Failed to send bulk notification', 'error', $e, [
                    'ticket_id' => $ticket->id,
                    'action' => $action,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    /**
     * Send ticket closed notification
     */
    public function sendTicketClosedNotification(Ticket $ticket): void
    {
        if (! config('tickets.notifications.enabled', true)) {
            return;
        }

        try {
            // Send to client
            if (config('tickets.notifications.send_to_client.ticket_closed', true)) {
                $client = $ticket->client;

                Email::to($client->email)
                    ->subject('[Ticket #'.$ticket->ticket_id.'] - Ticket Closed')
                    ->template('tickets::emails.client.ticket-closed', [
                        'client' => $client,
                        'ticket' => $ticket,
                        'login_url' => tenant_route('tenant.tickets.show', ['ticket' => $ticket->id]),
                    ])
                    ->send();
            }
        } catch (\Exception $e) {
            app_log('Failed to send ticket closed notification', 'error', $e, [
                'ticket_id' => $ticket->id,
                'error' => $e->getMessage(),
            ]);
        }
    }
}
