<?php

namespace App\Http\Controllers\Whatsapp;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Tenant\ManageChat;
use App\Models\Tenant\CampaignDetail;
use App\Models\Tenant\Chat;
use App\Models\Tenant\ChatMessage;
use App\Models\Tenant\Contact;
use App\Models\Tenant\MessageBot;
use App\Models\Tenant\TemplateBot;
use App\Services\pusher\PusherService;
use App\Traits\WhatsApp;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Netflie\WhatsAppCloudApi\WhatsAppCloudApi;
use stdClass;

class WhatsAppWebhookController extends Controller
{
    use WhatsApp;

    public $is_first_time = false;

    public $is_bot_stop = false;

    public $tenant_id = null;

    public $tenant_subdoamin = null;

    public $pusher_settings;

    /**
     * Handle incoming WhatsApp webhook requests
     */
    public function __invoke(Request $request)
    {
        // WhatsApp Webhook Verification
        if (isset($_GET['hub_mode']) && isset($_GET['hub_challenge']) && isset($_GET['hub_verify_token'])) {
            // Retrieve verify token from settings

            $settings = get_batch_settings(['whatsapp.webhook_verify_token']);
            $verifyToken = $settings['whatsapp.webhook_verify_token'];

            // Verify the webhook
            if ($_GET['hub_verify_token'] == $verifyToken && $_GET['hub_mode'] == 'subscribe') {
                // Directly output the challenge with proper headers
                header('Content-Type: text/plain');
                echo $_GET['hub_challenge'];
                exit;
            } else {
                // Send 403 Forbidden with a clear error message
                header('HTTP/1.1 403 Forbidden');
                header('Content-Type: text/plain');
                echo 'Verification failed: Invalid token or mode';
                exit;
            }
        }

        // Process webhook payload for messages and statuses
        $this->processWebhookPayload();

    }

    /**
     * Process incoming webhook payload
     */
    protected function processWebhookPayload()
    {
        $feedData = file_get_contents('php://input');

        if (! empty($feedData)) {
            $payload = json_decode($feedData, true);

            // Special ping message handling
            if (isset($payload['message']) && $payload['message'] === 'ctl_whatsmark_saas_ping' && isset($payload['identifier'])) {
                echo json_encode(['status' => true, 'message' => 'Webhook verified']);

                return;
            }

            $entry = reset($payload['entry']);
            $business_id = $entry['id'] ?? null;
            $phoneNumberId = $entry['changes'][0]['value']['metadata']['phone_number_id'] ?? null;
            $this->tenant_id = getTenantIdFromWhatsappDetails($business_id, $phoneNumberId);

            if (empty($this->tenant_id)) {
                return;
            }

            $this->pusher_settings = tenant_settings_by_group('pusher', $this->tenant_id);

            $this->tenant_subdoamin = tenant_subdomain_by_tenant_id($this->tenant_id);

            // Set the tenant ID in the trait for all subsequent API calls
            $this->setWaTenantId($this->tenant_id);

            whatsapp_log(
                'Webhook Payload Received',
                'info',
                [
                    'payload' => $feedData,
                    'tenant_id' => $this->tenant_id,
                ],
                null,
                $this->tenant_id
            );

            // Check for message ID to prevent duplicate processing
            $message_id = $payload['entry'][0]['changes'][0]['value']['messages'][0]['id'] ?? '';
            if (! empty($message_id)) {
                // Check if message already processed (similar to original code)
                $found = $this->checkMessageProcessed($message_id);
                if ($found) {
                    whatsapp_log(
                        'Duplicate Message Detected',
                        'warning',
                        [
                            'message_id' => $message_id,
                        ]
                    );

                    return;
                }
            }

            // Process the payload
            $this->processPayloadData($payload);

            // Forward webhook data if enabled
            $this->forwardWebhookData($feedData, $payload);
        }
    }

    /**
     * Check if message has already been processed
     */
    protected function checkMessageProcessed(string $messageId): bool
    {
        // Implement logic to check if message is already in database
        return \DB::table($this->tenant_subdoamin.'_chat_messages')
            ->where('message_id', $messageId)
            ->exists();
    }

    /**
     * Process payload data
     */
    protected function processPayloadData(array $payload)
    {
        whatsapp_log(
            'Processing Payload Data',
            'info',
            [
                'payload_entries' => count($payload['entry']),
                'tenant_id' => $this->tenant_id,
            ],
            null,
            $this->tenant_id
        );

        // Extract entry and changes
        $entry = array_shift($payload['entry']);
        $changes = array_shift($entry['changes']);
        $value = $changes['value'];

        // Process messages or statuses
        if (isset($value['messages'])) {
            $this->processIncomingMessages($value);
            $this->processBotSending($value);
        } elseif (isset($value['statuses'])) {
            $this->processMessageStatuses($value['statuses']);
        }
    }

    private function processBotSending(array $message_data)
    {
        if (! empty($message_data['messages'])) {
            $message = reset($message_data['messages']);
            $trigger_msg = isset($message['button']['text']) ? $message['button']['text'] : $message['text']['body'] ?? '';
            if (! empty($message['interactive']) && $message['interactive']['type'] == 'button_reply') {
                $trigger_msg = $message['interactive']['button_reply']['id'];
            }
            if (! empty($trigger_msg)) {
                $contact = reset($message_data['contacts']);
                $metadata = $message_data['metadata'];
            }

            try {
                $contact_number = $message['from'];
                $contact_data = $this->getContactData($contact_number, $contact['profile']['name']);
                if ($contact_data instanceof stdClass && empty((array) $contact_data)) {
                    return;
                }

                $query_trigger_msg = $trigger_msg;
                $reply_type = null;
                if ($this->is_first_time) {
                    $query_trigger_msg = '';
                    $reply_type = 3;
                }

                $current_interaction = Chat::fromTenant($this->tenant_subdoamin)->where([
                    'type' => $contact_data->type,
                    'type_id' => $contact_data->id,
                    'wa_no' => $message_data['metadata']['display_phone_number'],
                    'tenant_id' => $this->tenant_id,
                ])->first();

                if ($current_interaction->is_bots_stoped == 1 && (time() > strtotime($current_interaction->bot_stoped_time) + ((int) get_tenant_setting_by_tenant_id('whats-mark', 'restart_bots_after', null, $this->tenant_id) * 3600))) {
                    Chat::fromTenant($this->tenant_subdoamin)->where(['id' => $current_interaction->id, 'tenant_id' => $this->tenant_id])->update(['bot_stoped_time' => null, 'is_bots_stoped' => '0']);
                    $this->is_bot_stop = false;
                } elseif ($current_interaction->is_bots_stoped == 1) {
                    $this->is_bot_stop = true;
                }

                if (collect(get_tenant_setting_by_tenant_id('whats-mark', 'stop_bots_keyword', null, $this->tenant_id))->first(fn ($keyword) => str_contains($trigger_msg, $keyword))) {
                    Chat::fromTenant($this->tenant_subdoamin)->where(['id' => $current_interaction->id, 'tenant_id' => $this->tenant_id])->update(['bot_stoped_time' => date('Y-m-d H:i:s'), 'is_bots_stoped' => '1']);
                    $this->is_bot_stop = true;
                }

                if (! $this->is_bot_stop) {
                    // Fetch template and message bots based on interaction
                    $template_bots = TemplateBot::getTemplateBotsByRelType($contact_data->type ?? '', $query_trigger_msg, $this->tenant_id, $reply_type);
                    $message_bots = MessageBot::getMessageBotsbyRelType($contact_data->type ?? '', $query_trigger_msg, $this->tenant_id, $reply_type);

                    if (empty($template_bots) && empty($message_bots)) {
                        $template_bots = TemplateBot::getTemplateBotsByRelType($contact_data->type ?? '', $query_trigger_msg, $this->tenant_id, 4);
                        $message_bots = MessageBot::getMessageBotsbyRelType($contact_data->type ?? '', $query_trigger_msg, $this->tenant_id, 4);
                    }

                    $add_messages = function ($item) {
                        $item['header_message'] = $item['header_data_text'];
                        $item['body_message'] = $item['body_data'];
                        $item['footer_message'] = $item['footer_data'];

                        return $item;
                    };

                    $template_bots = array_map($add_messages, $template_bots);

                    // Iterate over template bots
                    foreach ($template_bots as $template) {
                        $template['rel_id'] = $contact_data->id;
                        if (! empty($contact_data->userid)) {
                            $template['userid'] = $contact_data->userid;
                        }

                        // Send template on exact match, contains, or first time
                        if (($template['reply_type'] == 1 && in_array(strtolower($trigger_msg), array_map('trim', array_map('strtolower', explode(',', $template['trigger']))))) || ($template['reply_type'] == 2 && ! empty(array_filter(explode(',', strtolower($template['trigger'])), fn ($word) => preg_match('/\b'.preg_quote(trim($word), '/').'\b/', strtolower($trigger_msg))))) || ($template['reply_type'] == 3 && $this->is_first_time) || $template['reply_type'] == 4) {
                            // Use the tenant ID when sending the template
                            $response = $this->setWaTenantId($this->tenant_id)->sendTemplate($contact_number, $template, 'template_bot', $metadata['phone_number_id']);

                            $chatId = $this->createOrUpdateInteraction($contact_number, $message_data['metadata']['display_phone_number'], $message_data['metadata']['phone_number_id'], $contact_data->firstname.' '.$contact_data->lastname, '', '', false);
                            $chatMessage = $this->storeBotMessages($template, $chatId, $contact_data, 'template_bot', $response);
                        }
                    }

                    // Iterate over message bots
                    foreach ($message_bots as $message) {
                        $message['rel_id'] = $contact_data->id;
                        if (! empty($contact_data->userid)) {
                            $message['userid'] = $contact_data->userid;
                        }
                        if (($message['reply_type'] == 1 && in_array(strtolower($trigger_msg), array_map('trim', array_map('strtolower', explode(',', $message['trigger']))))) || ($message['reply_type'] == 2 && ! empty(array_filter(explode(',', strtolower($message['trigger'])), fn ($word) => preg_match('/\b'.preg_quote(trim($word), '/').'\b/', strtolower($trigger_msg))))) || ($message['reply_type'] == 3 && $this->is_first_time) || $message['reply_type'] == 4) {
                            // Use the tenant ID when sending the message
                            $response = $this->setWaTenantId($this->tenant_id)
                                ->sendMessage($contact_number, $message, $metadata['phone_number_id']);

                            $chatId = $this->createOrUpdateInteraction($contact_number, $message_data['metadata']['display_phone_number'], $message_data['metadata']['phone_number_id'], $contact_data->firstname.' '.$contact_data->lastname, '', '', false);
                            $chatMessage = $this->storeBotMessages($message, $chatId, $contact_data, '', $response);
                        }
                    }
                }
            } catch (\Throwable $th) {
                whatsapp_log(
                    'Error processing bot sending',
                    'error',
                    [
                        'error' => $th->getMessage(),
                        'tenant_id' => $this->tenant_id,
                    ],
                    $th,
                    $this->tenant_id
                );
                file_put_contents(base_path().'/errors.json', json_encode([$th->getMessage()]));
            }
        }
    }

    /**
     * Process incoming messages
     */
    protected function processIncomingMessages(array $value)
    {
        $messageEntry = array_shift($value['messages']);
        $contact = array_shift($value['contacts']) ?? '';
        $name = $contact['profile']['name'] ?? '';
        $from = $messageEntry['from'];
        $metadata = $value['metadata'];
        $wa_no = $metadata['display_phone_number'];
        $wa_no_id = $metadata['phone_number_id'];
        $messageType = $messageEntry['type'];
        $message_id = $messageEntry['id'];
        $ref_message_id = isset($messageEntry['context']) ? $messageEntry['context']['id'] : '';

        // Determine if this is a first-time interaction
        $this->is_first_time = $this->isFirstTimeInteraction($from);

        // Extract message content based on type
        $message = $this->extractMessageContent($messageEntry, $messageType);
        if ($messageType == 'image' || $messageType == 'audio' || $messageType == 'document' || $messageType == 'video') {
            $media_id = $messageEntry[$messageType]['id'];
            // Make sure to use setWaTenantId when retrieving URL
            $attachment = $this->setWaTenantId($this->tenant_id)->retrieveUrl($media_id);
        }

        whatsapp_log(
            'Processing Incoming Message',
            'info',
            [
                'from' => $from,
                'name' => $name,
                'message_type' => $messageType,
                'is_first_time' => $this->is_first_time,
                'tenant_id' => $this->tenant_id,
            ],
            null,
            $this->tenant_id
        );

        // Create or update interaction
        $interaction_id = $this->createOrUpdateInteraction(
            $from,
            $wa_no,
            $wa_no_id,
            $name,
            $message,
            $messageType
        );

        // Store interaction message
        $message_id = $this->storeInteractionMessage(
            $interaction_id,
            $from,
            $message_id,
            $message,
            $messageType,
            $ref_message_id,
            $metadata,
            $attachment ?? ''
        );

        Chat::fromTenant($this->tenant_subdoamin)->where('id', $interaction_id)->update([
            'last_message' => $message,
            'last_msg_time' => now(),
            'updated_at' => now(),
        ]);

        if (
            ! empty($this->pusher_settings['app_key']) && ! empty($this->pusher_settings['app_secret']) && ! empty($this->pusher_settings['app_id']) && ! empty($this->pusher_settings['cluster'])
        ) {

            $pusherService = new PusherService($this->tenant_id);
            $pusherService->trigger('whatsmark-saas-chat-channel', 'whatsmark-saas-chat-event', [
                'chat' => ManageChat::newChatMessage($interaction_id, $message_id, $this->tenant_id),
            ]);
        }
    }

    /**
     * Check if this is a first-time interaction
     */
    protected function isFirstTimeInteraction(string $from): bool
    {
        return ! (bool) Chat::fromTenant($this->tenant_subdoamin)->where('receiver_id', $from)->count();
    }

    /**
     * Extract message content based on type
     */
    protected function extractMessageContent(array $messageEntry, string $messageType): string
    {
        switch ($messageType) {
            case 'text':
                return $messageEntry['text']['body'] ?? '';
            case 'interactive':
                return $messageEntry['interactive']['button_reply']['title'] ?? '';
            case 'button':
                return $messageEntry['button']['text'] ?? '';
            case 'reaction':
                return json_decode('"'.($messageEntry['reaction']['emoji'] ?? '').'"', false, 512, JSON_UNESCAPED_UNICODE);
            case 'image':
            case 'audio':
            case 'document':
            case 'video':
                return $messageType;
            default:
                return 'Unknown message type';
        }
    }

    /**
     * Create or update interaction
     */
    protected function createOrUpdateInteraction(
        string $from,
        string $wa_no,
        string $wa_no_id,
        string $name,
        string $message,
        string $messageType,
        bool $enableTime = true
    ): int {
        // Retrieve contact data (similar to original implementation)
        $contact_data = $this->getContactData($from, $name);

        // Check if a record with the same receiver_id exists
        $existingChat = Chat::fromTenant($this->tenant_subdoamin)->where('tenant_id', $this->tenant_id)->where('receiver_id', $from)->first();

        if ($existingChat) {

            Chat::fromTenant($this->tenant_subdoamin)->where('id', $existingChat->id)->update([
                'wa_no' => $wa_no,
                'wa_no_id' => $wa_no_id,
                'name' => $name,
                'last_message' => $message,
                'time_sent' => now(),
                'type' => $contact_data->type ?? 'guest',
                'type_id' => $contact_data->id ?? '',
                'updated_at' => now(),
                'tenant_id' => $this->tenant_id,
            ] + ($enableTime ? ['time_sent' => now()] : []));

            return $existingChat->id;
        } else {

            $conversationType = $contact_data->type ?? 'guest';
            $featureService = app(\App\Services\FeatureService::class);

            // Create new chat first to get the chat ID for guest type
            $newChatId = Chat::fromTenant($this->tenant_subdoamin)->insertGetId([
                'receiver_id' => $from,
                'wa_no' => $wa_no,
                'wa_no_id' => $wa_no_id,
                'name' => $name,
                'last_message' => $message,
                'agent' => json_encode(['assign_id' => $contact_data->assigned_id ?? 0, 'agents_id' => '']),
                'time_sent' => now(),
                'type' => $conversationType,
                'type_id' => $contact_data->id ?? '',
                'created_at' => now(),
                'updated_at' => now(),
                'tenant_id' => $this->tenant_id,
            ]);

            // Check conversation limit for ALL types (lead, customer, guest)
            if (in_array($conversationType, ['lead', 'customer', 'guest'])) {
                // For guest type, use the new chat ID; for others, use contact ID
                $identifierForCheck = ($conversationType === 'guest') ? $newChatId : ($contact_data->id ?? '');

                if (! empty($identifierForCheck)) {
                    if ($featureService->checkConversationLimit($identifierForCheck, $this->tenant_id, $this->tenant_subdoamin, $conversationType)) {
                        // Log the limit but don't block incoming messages (customer service)
                        whatsapp_log('Conversation limit reached for new interaction', 'warning', [
                            'tenant_id' => $this->tenant_id,
                            'contact_or_chat_id' => $identifierForCheck,
                            'contact_type' => $conversationType,
                            'from' => $from,
                        ], null, $this->tenant_id);
                    } else {
                        // Track new conversation usage
                        $featureService->trackNewConversation($identifierForCheck, $this->tenant_id, $this->tenant_subdoamin, $conversationType);
                    }
                }
            }

            return $newChatId;
        }
    }

    /**
     * Store interaction message
     */
    protected function storeInteractionMessage(
        int $interaction_id,
        string $from,
        string $message_id,
        string $message,
        string $messageType,
        string $ref_message_id,
        array $metadata,
        string $url = ''
    ) {
        return ChatMessage::fromTenant($this->tenant_subdoamin)->insertGetId([
            'interaction_id' => $interaction_id,
            'sender_id' => $from,
            'message_id' => $message_id,
            'message' => $message,
            'type' => $messageType,
            'staff_id' => null,
            'status' => 'sent',
            'time_sent' => now(),
            'ref_message_id' => $ref_message_id,
            'created_at' => now(),
            'updated_at' => now(),
            'url' => $url,
            'tenant_id' => $this->tenant_id,
        ]);
    }

    /**
     * Process message statuses
     */
    protected function processMessageStatuses(array $statuses)
    {
        foreach ($statuses as $status) {
            $id = $status['id'];
            $status_value = $status['status'];

            $status_message = null;
            $errors = $status['errors'] ?? [];

            $error_data = array_column($errors, 'error_data');
            $details = array_column($error_data, 'details');

            $status_message = reset($details) ?: null;

            // Update chat message
            CampaignDetail::where('whatsapp_id', $id)->update(['message_status' => $status_value, 'response_message' => $status_message]);
            $message = ChatMessage::fromTenant($this->tenant_subdoamin)->where('message_id', $id)->first();

            if ($message) {
                $message->update([
                    'status' => $status_value,
                    'status_message' => $status_message,
                    'updated_at' => now(),
                ]);

                if (
                    ! empty($this->pusher_settings['app_key']) && ! empty($this->pusher_settings['app_secret']) && ! empty($this->pusher_settings['app_id']) && ! empty($this->pusher_settings['cluster'])
                ) {

                    $pusherService = new PusherService($this->tenant_id);
                    $pusherService->trigger('whatsmark-saas-chat-channel', 'whatsmark-saas-chat-event', [
                        'chat' => ManageChat::newChatMessage($$message->interaction_id, $message->id, $this->tenant_id),
                    ]);
                }
            }
        }
    }

    /**
     * Forward webhook data if enabled
     */
    protected function forwardWebhookData(string $feedData, array $payload)
    {
        $enabled = get_tenant_setting_by_tenant_id('whats-mark', 'enable_webhook_resend', '', $this->tenant_id);
        $url = get_tenant_setting_by_tenant_id('whats-mark', 'whatsapp_data_resend_to', '', $this->tenant_id);
        $method = strtoupper(get_tenant_setting_by_tenant_id('whats-mark', 'webhook_resend_method', 'POST', $this->tenant_id));

        if ($enabled && filter_var($url, FILTER_VALIDATE_URL)) {
            try {
                switch ($method) {
                    case 'POST':
                        $response = \Http::post($url, $payload);
                        break;
                    case 'GET':
                        $response = \Http::get($url, $payload);
                        break;
                    default:
                        $response = \Http::withBody($feedData, 'application/json')->send($method, $url);
                        break;
                }
            } catch (\Exception $e) {
                whatsapp_log('Webhook Forward Error', 'error', [
                    'message' => $e->getMessage(),
                ], $e);
            }
        }
    }

    /**
     * Get contact data (placeholder method)
     */
    protected function getContactData(string $from, string $name): object
    {
        $contact = Contact::fromTenant($this->tenant_subdoamin)->where('tenant_id', $this->tenant_id)
            ->where(function ($query) use ($from) {
                $query->where('phone', $from)
                    ->orWhere('phone', '+'.$from);
            })
            ->first();
        if ($contact) {
            return $contact;
        }
        if (get_tenant_setting_by_tenant_id('whats-mark', 'auto_lead_enabled', null, $this->tenant_id)) {
            $name = explode(' ', $name);
            $contact = Contact::fromTenant($this->tenant_subdoamin)->create([
                'firstname' => $name[0],
                'lastname' => count($name) > 1 ? implode(' ', array_slice($name, 1)) : '',
                'type' => 'lead',
                'phone' => $from[0] === '+' ? $from : '+'.$from,
                'assigned_id' => get_tenant_setting_by_tenant_id('whats-mark', 'lead_assigned_to', null, $this->tenant_id),
                'status_id' => get_tenant_setting_by_tenant_id('whats-mark', 'lead_status', null, $this->tenant_id),
                'source_id' => get_tenant_setting_by_tenant_id('whats-mark', 'lead_source', null, $this->tenant_id),
                'addedfrom' => '0',
                'tenant_id' => $this->tenant_id,
            ]);

            return $contact;
        }

        return (object) [];
    }

    public function storeBotMessages($data, $interactionId, $relData, $type, $response)
    {
        $data['sending_count'] = (int) $data['sending_count'] + 1;

        if ($type == 'template_bot') {
            $header = parseText($data['rel_type'], 'header', $data);
            $body = parseText($data['rel_type'], 'body', $data);
            $footer = parseText($data['rel_type'], 'footer', $data);

            $buttonHtml = '';
            if (! empty(json_decode($data['buttons_data']))) {
                $buttons = json_decode($data['buttons_data']);
                $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
                foreach ($buttons as $button) {
                    $buttonHtml .= "<button class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
                        dark:bg-gray-800 dark:text-green-400'>".e($button->text).'</button>';
                }
                $buttonHtml .= '</div>';
            }

            $headerData = '';
            $fileExtensions = get_meta_allowed_extension();
            $extension = strtolower(pathinfo($data['filename'], PATHINFO_EXTENSION));
            $fileType = array_key_first(array_filter($fileExtensions, fn ($data) => in_array('.'.$extension, explode(', ', $data['extension']))));
            if ($data['header_data_format'] === 'IMAGE' && $fileType == 'image') {
                $headerData = "<a href='".asset('storage/'.$data['filename'])."' data-lightbox='image-group'>
                <img src='".asset('storage/'.$data['filename'])."' class='rounded-lg w-full mb-2'>
            </a>";
            } elseif ($data['header_data_format'] === 'TEXT' || $data['header_data_format'] === '') {
                $headerData = "<span class='font-bold mb-3'>".nl2br(decodeWhatsAppSigns(e($header ?? ''))).'</span>';
            } elseif ($data['header_data_format'] === 'DOCUMENT') {
                $headerData = "<a href='".asset('storage/'.$data['filename'])."' target='_blank' class='btn btn-secondary w-full'>".t('document').'</a>';
            } elseif ($data['header_data_format'] === 'VIDEO') {
                $headerData = "<video src='".asset('storage/'.$data['filename'])."' controls class='rounded-lg w-full'></video>";
            }

            TemplateBot::where(['id' => $data['id'], 'tenant_id' => $this->tenant_id])->update(['sending_count' => $data['sending_count'] + 1]);

            $chat_message = [
                'interaction_id' => $interactionId,
                'sender_id' => get_tenant_setting_by_tenant_id('whatsapp', 'wm_default_phone_number', null, $this->tenant_id),
                'url' => null,
                'message' => "
                $headerData
                <p>".nl2br(decodeWhatsAppSigns(e($body)))."</p>
                <span class='text-gray-500 text-sm'>".nl2br(decodeWhatsAppSigns(e($footer ?? '')))."</span>
                $buttonHtml
            ",
                'status' => 'sent',
                'time_sent' => now()->toDateTimeString(),
                'message_id' => $response['data']->messages[0]->id ?? null,
                'staff_id' => 0,
                'type' => 'text',
                'tenant_id' => $this->tenant_id,
                'is_read' => '1',
            ];

            $message_id = ChatMessage::fromTenant($this->tenant_subdoamin)->insertGetId($chat_message);

            if (
                ! empty($this->pusher_settings['app_key']) && ! empty($this->pusher_settings['app_secret']) && ! empty($this->pusher_settings['app_id']) && ! empty($this->pusher_settings['cluster'])
            ) {
                $pusherService = new PusherService($this->tenant_id);
                $pusherService->trigger('whatsmark-saas-chat-channel', 'whatsmark-saas-chat-event', [
                    'chat' => ManageChat::newChatMessage($interactionId, $message_id, $this->tenant_id),
                ]);
            }

            return $message_id;
        }

        $type = $type === 'flow' ? 'flow' : 'bot_files';
        $data = parseMessageText($data);
        $header = $data['bot_header'] ?? '';
        $body = $data['reply_text'] ?? '';
        $footer = $data['bot_footer'] ?? '';

        $headerImage = '';
        $allowedExtensions = get_meta_allowed_extension();

        $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
        $option = false;

        if (! empty($data['button1_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-green-400'>".e($data['button1']).'</button>';
            $option = true;
        }
        if (! empty($data['button2_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-green-400'>".e($data['button2']).'</button>';
            $option = true;
        }
        if (! empty($data['button3_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-green-400'>".e($data['button3']).'</button>';
            $option = true;
        }
        if (! $option && ! empty($data['button_name']) && ! empty($data['button_url']) && filter_var($data['button_url'], FILTER_VALIDATE_URL)) {
            $buttonHtml .= "<a href='".e($data['button_url'])."' class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-green-400 mt-2'> <svg class='w-4 h-4 text-green-500' aria-hidden='true' xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24'> <path stroke='currentColor' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M18 14v4.833A1.166 1.166 0 0 1 16.833 20H5.167A1.167 1.167 0 0 1 4 18.833V7.167A1.166 1.166 0 0 1 5.167 6h4.618m4.447-2H20v5.768m-7.889 2.121 7.778-7.778'/> </svg><span class='whitespace-nowrap'>".e($data['button_name']).'</a>';
            $option = true;
        }

        $extension = strtolower(pathinfo($data['filename'], PATHINFO_EXTENSION));
        $fileType = array_key_first(array_filter($allowedExtensions, fn ($data) => in_array('.'.$extension, explode(', ', $data['extension']))));
        if (! $option && ! empty($data['filename']) && $fileType == 'image') {
            $headerImage = "<a href='".asset('storage/'.$data['filename'])."' data-lightbox='image-group'>
            <img src='".asset('storage/'.$data['filename'])."' class='rounded-lg w-full mb-2'>
        </a>";
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'document') {
            $headerImage = "<a href='".asset('storage/'.$data['filename'])."' target='_blank' class='bg-gray-100 text-green-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-green-400'>".t('document').'</a>';
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'video') {
            $headerImage = "<video src='".asset('storage/'.$data['filename'])."' controls class='rounded-lg w-full'></video>";
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'audio') {
            $headerImage = "<audio controls class='w-64'><source src='".asset('storage/'.$data['filename'])."' type='audio/mpeg'></audio>";
        }
        $buttonHtml .= '</div>';

        MessageBot::where('id', $data['id'])->update(['sending_count' => $data['sending_count'] + 1]);

        $buttondata = $buttonHtml == "<div class='flex flex-col mt-2 space-y-2'></div>" ? '' : $buttonHtml;

        $chat_message = [
            'interaction_id' => $interactionId,
            'sender_id' => get_tenant_setting_by_tenant_id('whatsapp', 'wm_default_phone_number', null, $this->tenant_id),
            'url' => null,
            'message' => $headerImage."
            <span class='font-bold mb-3'>".nl2br(e($header ?? '')).'</span>
            <p>'.nl2br(decodeWhatsAppSigns(e($body)))."</p>
            <span class='text-gray-500 text-sm'>".nl2br(e($footer ?? ''))."</span>
            $buttondata
        ",
            'status' => 'sent',
            'time_sent' => now()->toDateTimeString(),
            'message_id' => $response['data']->messages[0]->id ?? null,
            'staff_id' => 0,
            'type' => 'text',
            'tenant_id' => $this->tenant_id,
            'is_read' => '1',
        ];

        $message_id = ChatMessage::fromTenant($this->tenant_subdoamin)->insertGetId($chat_message);

        if (
            ! empty($this->pusher_settings['app_key']) && ! empty($this->pusher_settings['app_secret']) && ! empty($this->pusher_settings['app_id']) && ! empty($this->pusher_settings['cluster'])
        ) {
            $pusherService = new PusherService($this->tenant_id);
            $pusherService->trigger('whatsmark-saas-chat-channel', 'whatsmark-saas-chat-event', [
                'chat' => ManageChat::newChatMessage($interactionId, $message_id, $this->tenant_id),
            ]);
        }

        return $message_id;
    }

    /**
     * Send a message via WhatsApp
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function send_message(Request $request, $subdomain)
    {
        $this->tenant_subdoamin = $subdomain;

        try {
            // Get request data
            $id = $request->input('id', '');
            $type = $request->input('type');
            $type_id = $request->input('type_id');

            // Find existing chat/interaction
            $query = Chat::fromTenant($this->tenant_subdoamin);
            if (! empty($type_id)) {
                $query->where('type', $type)
                    ->where('type_id', $type_id);
            }
            $existing_interaction = $query->where('id', $id)->first();

            if (! $existing_interaction) {
                return response()->json(['error' => 'Interaction not found'], 404);
            }

            $this->tenant_id = $existing_interaction->tenant_id;
            $this->setWaTenantId($this->tenant_id);

            // CONVERSATION LIMIT LOGIC - BEFORE SENDING
            $conversationTrackingNeeded = false;
            $identifierForTracking = null;

            if (in_array($type, ['customer', 'lead', 'guest'])) {
                $featureService = app(\App\Services\FeatureService::class);

                // Force initialize conversation tracking
                $featureService->forceInitializeConversationTracking();

                // For guest type, use chat ID instead of contact ID
                $identifierForCheck = ($type === 'guest') ? $id : $type_id;

                // Check if this would be a new conversation
                $conversationTrackingNeeded = $this->shouldTrackNewConversation(
                    $identifierForCheck,
                    $type,
                    $this->tenant_id,
                    $this->tenant_subdoamin
                );

                if ($conversationTrackingNeeded) {
                    $identifierForTracking = $identifierForCheck;

                    // Check conversation limit before sending
                    if ($featureService->checkConversationLimit($identifierForCheck, $this->tenant_id, $this->tenant_subdoamin, $type)) {
                        whatsapp_log('DEBUG: Conversation limit reached - BLOCKING MESSAGE', 'warning', [
                            'identifier' => $identifierForCheck,
                            'type' => $type,
                            'current_usage' => $featureService->getCurrentUsage('conversations'),
                            'current_limit' => $featureService->getLimit('conversations'),
                        ]);

                        return response()->json([
                            'success' => false,
                            'error' => 'Conversation limit reached. Please upgrade your plan to continue messaging.',
                            'limit_reached' => true,
                        ], 429);
                    }
                }

            }

            $to = $existing_interaction->receiver_id;
            $message = strip_tags($request->input('message', ''));

            // Parse message text for contacts or leads
            $user_id = null;
            if ($type == 'customer' || $type == 'lead') {
                $contact = Contact::fromTenant($this->tenant_subdoamin)->find($type_id);
                $user_id = $contact->user_id ?? null;
            }

            $message_data = parseMessageText([
                'rel_type' => $type,
                'rel_id' => $type_id,
                'reply_text' => $message,
                'userid' => $user_id,
                'tenant_id' => $this->tenant_id,
            ]);

            $message = $message_data['reply_text'] ?? $message;
            $ref_message_id = $request->input('ref_message_id');
            $message_data = [];

            // Add text message if provided
            if (! empty($message)) {
                $message_data[] = [
                    'type' => 'text',
                    'text' => [
                        'preview_url' => true,
                        'body' => $message,
                    ],
                ];
            }

            // Handle file attachments (using existing method)
            $attachments = [
                'audio' => $request->file('audio'),
                'image' => $request->file('image'),
                'video' => $request->file('video'),
                'document' => $request->file('document'),
            ];

            foreach ($attachments as $type => $file) {
                if (! empty($file)) {
                    $file_url = $this->handle_attachment_upload($file);

                    $message_data[] = [
                        'type' => $type,
                        $type => [
                            'url' => url('storage/whatsapp-attachments/'.$file_url),
                        ],
                    ];
                }
            }

            if (empty($message_data)) {
                return response()->json(['error' => 'No message content provided'], 400);
            }

            // Send WhatsApp messages (using existing WhatsAppCloudApi)
            $whatsapp_success = false;
            $messageIds = [];

            // Initialize WhatsApp Cloud API client
            $whatsapp_cloud_api = new \Netflie\WhatsAppCloudApi\WhatsAppCloudApi([
                'from_phone_number_id' => $existing_interaction->wa_no_id,
                'access_token' => $this->setWaTenantId($this->tenant_id)->getToken(),
            ]);

            try {
                foreach ($message_data as $data) {
                    $response = null;

                    switch ($data['type']) {
                        case 'text':
                            $response = $whatsapp_cloud_api->sendTextMessage($to, $data['text']['body']);
                            break;
                        case 'audio':
                            $response = $whatsapp_cloud_api->sendAudio($to, new \Netflie\WhatsAppCloudApi\Message\Media\LinkID($data['audio']['url']));
                            break;
                        case 'image':
                            $response = $whatsapp_cloud_api->sendImage($to, new \Netflie\WhatsAppCloudApi\Message\Media\LinkID($data['image']['url']));
                            break;
                        case 'video':
                            $response = $whatsapp_cloud_api->sendVideo($to, new \Netflie\WhatsAppCloudApi\Message\Media\LinkID($data['video']['url']));
                            break;
                        case 'document':
                            $fileName = basename($data['document']['url']);
                            $response = $whatsapp_cloud_api->sendDocument($to, new \Netflie\WhatsAppCloudApi\Message\Media\LinkID($data['document']['url']), $fileName, '');
                            break;
                        default:
                            continue 2;
                    }

                    // Decode the response JSON
                    $response_data = $response->decodedBody();

                    // Store the message ID if available
                    if (isset($response_data['messages'][0]['id'])) {
                        $messageIds[] = $response_data['messages'][0]['id'];
                        $whatsapp_success = true;

                    }
                }
            } catch (\Exception $e) {
                whatsapp_log('Exception during WhatsApp send', 'error', [
                    'to' => $to,
                    'error' => $e->getMessage(),
                ], $e, $this->tenant_id);
            }

            // POST-SEND PROCESSING - CRITICAL FIX
            if ($whatsapp_success) {

                // 1. Update chat record with last message time
                $chatUpdated = $this->updateChatAfterOutgoingMessage(
                    $id,
                    $message,
                    $this->tenant_id,
                    $this->tenant_subdoamin
                );

                // 2. Track conversation if needed (NEW CONVERSATION)
                if ($conversationTrackingNeeded && $identifierForTracking) {
                    $conversationTracked = $this->trackConversationAfterSend(
                        $identifierForTracking,
                        $type,
                        $this->tenant_id,
                        $this->tenant_subdoamin
                    );

                }

                // Create or update chat entry
                $interaction_id = $this->createOrUpdateInteraction($to, $existing_interaction->wa_no, $existing_interaction->wa_no_id, $existing_interaction->name, $message ?? 'Media message', '', false);

                // Save messages to database
                foreach ($message_data as $index => $data) {
                    $message_id = ChatMessage::fromTenant($this->tenant_subdoamin)->insertGetId([
                        'interaction_id' => $interaction_id,
                        'sender_id' => $existing_interaction->wa_no,
                        'message' => $message,
                        'message_id' => $messageIds[$index] ?? null,
                        'type' => $data['type'] ?? 'text',
                        'staff_id' => auth()->id(),
                        'url' => isset($data[$data['type']]['url']) ? basename($data[$data['type']]['url']) : null,
                        'status' => 'sent',
                        'time_sent' => now(),
                        'ref_message_id' => $ref_message_id ?? '',
                        'created_at' => now(),
                        'updated_at' => now(),
                        'is_read' => 1,
                        'tenant_id' => $this->tenant_id,
                        'is_read' => '1',
                    ]);

                    // Broadcast message via Pusher if enabled
                    if (
                        ! empty(get_tenant_setting_by_tenant_id('pusher', 'app_key', null, $this->tenant_id)) && ! empty(get_tenant_setting_by_tenant_id('pusher', 'app_secret', null, $this->tenant_id)) && ! empty(get_tenant_setting_by_tenant_id('pusher', 'app_id', null, $this->tenant_id)) && ! empty(get_tenant_setting_by_tenant_id('pusher', 'cluster', null, $this->tenant_id))
                    ) {
                        $pusherService = new PusherService($this->tenant_id);
                        $pusherService->trigger('whatsmark-saas-chat-channel', 'whatsmark-saas-chat-event', [
                            'chat' => ManageChat::newChatMessage($interaction_id, $message_id, $this->tenant_id),
                        ]);
                    }
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Message sent successfully',
                    'conversation_tracked' => $conversationTrackingNeeded ? ($conversationTracked ?? false) : false,
                    'chat_updated' => $chatUpdated ?? false,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send WhatsApp message',
                ], 500);
            }
        } catch (\Exception $e) {
            whatsapp_log('send_message exception', 'error', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ], $e, $this->tenant_id);

            return response()->json([
                'success' => false,
                'message' => 'Internal server error: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Handle file attachment uploads
     *
     * @param  \Illuminate\Http\UploadedFile  $file
     * @return string The stored file name
     */
    protected function handle_attachment_upload($file)
    {
        if (empty($file)) {
            return null;
        }

        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $extension = $file->getClientOriginalExtension();

        // Sanitize filename: remove special chars, replace spaces
        $cleanName = Str::slug($originalName, '_'); // e.g., "WhatsApp_Video_2025_02_20_at_13_06_57_08d1199a"

        // Append timestamp to ensure uniqueness
        $fileName = time().'_'.$cleanName.'.'.$extension;

        // Store the file
        $file->storeAs('whatsapp-attachments', $fileName, 'public');

        return $fileName;
    }

    /**
     * Update chat record after tenant sends outgoing message
     */
    protected function updateChatAfterOutgoingMessage($chatId, $message, $tenantId, $tenantSubdomain)
    {
        try {
            $updated = Chat::fromTenant($tenantSubdomain)
                ->where('id', $chatId)
                ->where('tenant_id', $tenantId)
                ->update([
                    'last_message' => strip_tags($message),
                    'time_sent' => now(),
                    'updated_at' => now(),
                ]);

            return $updated > 0;
        } catch (\Exception $e) {
            whatsapp_log('Failed to update chat after outgoing message', 'error', [
                'chat_id' => $chatId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage(),
            ], $e, $tenantId);

            return false;
        }
    }

    /**
     * Check if conversation tracking is needed for outgoing message
     */
    protected function shouldTrackNewConversation($identifier, $type, $tenantId, $tenantSubdomain)
    {
        try {
            $featureService = app(\App\Services\FeatureService::class);

            // Check if there's an active session
            $hasActiveSession = $featureService->isConversationSessionActive(
                $identifier,
                $tenantId,
                $tenantSubdomain,
                $type
            );

            return ! $hasActiveSession; // Track if no active session
        } catch (\Exception $e) {
            whatsapp_log('Error checking conversation tracking need', 'error', [
                'identifier' => $identifier,
                'type' => $type,
                'error' => $e->getMessage(),
            ], $e, $tenantId);

            return false;
        }
    }

    /**
     * Track conversation after successful outgoing message
     */
    protected function trackConversationAfterSend($identifier, $type, $tenantId, $tenantSubdomain)
    {
        try {
            $featureService = app(\App\Services\FeatureService::class);

            $tracked = $featureService->trackNewConversation(
                $identifier,
                $tenantId,
                $tenantSubdomain,
                $type
            );

            return $tracked;
        } catch (\Exception $e) {
            whatsapp_log('Failed to track conversation after send', 'error', [
                'identifier' => $identifier,
                'type' => $type,
                'error' => $e->getMessage(),
            ], $e, $tenantId);

            return false;
        }
    }
}
