<?php

namespace App\Livewire\Tenant\Tables;

use App\Enum\Tenant\WhatsAppTemplateRelationType;
use App\Facades\TenantCache;
use App\Models\Tenant\Contact;
use App\Models\Tenant\Source;
use App\Models\Tenant\Status;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\On;
use PowerComponents\LivewirePowerGrid\Button;
use PowerComponents\LivewirePowerGrid\Column;
use PowerComponents\LivewirePowerGrid\Components\SetUp\Exportable;
use PowerComponents\LivewirePowerGrid\Facades\Filter;
use PowerComponents\LivewirePowerGrid\Facades\PowerGrid;
use PowerComponents\LivewirePowerGrid\PowerGridComponent;
use PowerComponents\LivewirePowerGrid\PowerGridFields;
use PowerComponents\LivewirePowerGrid\Traits\WithExport;

final class ContactTable extends PowerGridComponent
{
    use WithExport;

    public string $tableName = 'contact-table';

    public bool $deferLoading = false;

    public bool $showFilters = false;

    public string $sortField = 'created_at';

    public string $sortDirection = 'desc';

    public string $loadingComponent = 'components.custom-loading';

    public array $selected = [];

    public $tenant_id;

    public $tenant_subdomain;

    protected const CACHE_KEY_USERS = 'contacts_table_users_for_filter';

    protected const CACHE_KEY_STATUSES = 'contacts_table_statuses_for_filter';

    protected const CACHE_KEY_SOURCES = 'contacts_table_sources_for_filter';

    protected const CACHE_DURATION = 600;

    public function boot(): void
    {
        config(['livewire-powergrid.filter' => 'outside']);
        $this->tenant_id = tenant_id();
        $this->tenant_subdomain = tenant_subdomain_by_tenant_id($this->tenant_id);
    }

    public function setUp(): array
    {
        $this->showCheckBox();

        return [
            PowerGrid::exportable('contacts-list')
                ->striped()
                ->stripTags(true)
                ->type(Exportable::TYPE_XLS, Exportable::TYPE_CSV),

            PowerGrid::header()
                ->showToggleColumns()
                ->withoutLoading()
                ->showSearchInput(),

            PowerGrid::footer()
                ->showPerPage()
                ->showRecordCount(),
        ];
    }

    public function header(): array
    {
        $buttons = [];

        $alternativePositionClass = 'absolute md:top-[4.5rem] sm:top-[4.75rem] top-[6.5rem] xs:top-[6.5rem] left-[166px] lg:left-[182px] md:left-[211px] sm:left-[207px]';
        $buttonClass = $alternativePositionClass;

        $contactCount = Contact::fromTenant($this->tenant_subdomain)->count();

        if (checkPermission('tenant.contact.delete')) {
            if ($contactCount > 0) {
                $buttons[] = Button::add('bulk-delete')
                    ->id()
                    ->slot(t('bulk_delete').'(<span x-text="window.pgBulkActions.count(\''.$this->tableName.'\')"></span>)')
                    ->class("iinline-flex items-center justify-center px-3 py-2 text-sm border border-transparent rounded-md font-medium disabled:opacity-50 disabled:pointer-events-none transition bg-red-600 text-white hover:bg-red-500 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-600 whitespace-nowrap $buttonClass")
                    ->dispatch('bulkDelete.'.$this->tableName, []);
            }
        }

        return $buttons;
    }

    public function datasource(): Builder
    {
        $query = Contact::fromTenant($this->tenant_subdomain)
            ->where('tenant_id', $this->tenant_id)
            ->selectRaw('*, ROW_NUMBER() OVER (ORDER BY id) as row_num')
            ->with([
                'user:id,firstname,lastname,avatar',
                'status:id,name,color',
                'source:id,name',
            ]);

        // Filter contacts based on permissions
        if (checkPermission('tenant.contact.view')) {
            return $query; // all contacts
        } elseif (checkPermission('tenant.contact.view_own')) {
            $user = auth()->user();
            if ($user->user_type === 'tenant' && $user->tenant_id === tenant_id() && $user->is_admin === false) {
                $staffId = $user->id;
            }

            return $query->where('assigned_id', $staffId); // show only their assigned contacts
        }

        // Default return if no permissions match
        return $query;
    }

    public function relationSearch(): array
    {
        return [
            'user' => [
                'firstname',
                'lastname',
            ],
            'status' => [
                'name',
            ],
            'source' => [
                'name',
            ],
        ];
    }

    public function fields(): PowerGridFields
    {
        return PowerGrid::fields()
            ->add('row_num')
            ->add('firstname', function ($contact) {
                return view('components.contacts.name-with-actions', [
                    'id' => $contact->id,
                    'fullName' => $contact->firstname.' '.$contact->lastname,
                ])->render();
            })
            ->add('firstname_raw', fn ($contact) => e($contact->firstname.' '.$contact->lastname))
            ->add(
                'status_id',
                function ($contact) {
                    if (! empty($contact->status->color)) {
                        return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium"
                        style="background-color: '.e($contact->status->color).'20; color: '.e($contact->status->color).';">
                        '.e($contact->status->name).'</span>';
                    }
                }
            )
            ->add(
                'assigned_id',
                function ($contact) {
                    if (! $contact->user) {
                        return t('not_assigned');
                    }

                    $profileImage = ! empty($contact->user->avatar) && Storage::disk('public')->exists($contact->user->avatar)
                        ? Storage::url($contact->user->avatar)
                        : asset('img/user-placeholder.jpg');

                    $fullName = e($contact->user->firstname.' '.$contact->user->lastname);

                    return '<div class="relative group flex items-center cursor-pointer">
                        <a href="'.tenant_route('tenant.staff.details', ['staffId' => $contact->assigned_id]).'">
                            <img src="'.$profileImage.'"
                                class="w-9 h-9 rounded-full mx-3 object-cover"
                                data-tippy-content="'.$fullName.'">
                        </a>
                    </div>';
                }
            )
            ->add('source_id', fn ($contact) => $contact->source?->name ?? 'N/A')
            ->add('created_at_formatted', function ($contact) {
                return '<div class="relative group">
                        <span class="cursor-default" data-tippy-content="'.format_date_time($contact->created_at).'">'
                    .Carbon::parse($contact->created_at)->diffForHumans(['options' => Carbon::JUST_NOW]).'</span>
                    </div>';
            })
            ->add('type', function ($contact) {
                return t($contact->type);
            });
    }

    public function columns(): array
    {
        return [
            Column::make(t('SR.NO'), 'row_num')
                ->sortable(),

            Column::make(t('name'), 'firstname')
                ->bodyAttribute('class="relative"')
                ->sortable()
                ->searchable()
                ->visibleInExport(false),

            Column::make(t('name'), 'firstname_raw')
                ->hidden()
                ->visibleInExport(true),

            Column::make(t('type'), 'type')
                ->sortable()
                ->searchable(),

            Column::make(t('phone'), 'phone')
                ->sortable()
                ->searchable(),

            Column::make(t('assigned'), 'assigned_id')
                ->sortable()
                ->searchable(),

            Column::make(t('status'), 'status_id')
                ->sortable()
                ->searchable(),

            Column::make(t('source'), 'source_id')
                ->sortable()
                ->searchable(),

            Column::make(t('active'), 'is_enabled')
                ->sortable()
                ->toggleable(hasPermission: true, trueLabel: '1', falseLabel: '0')
                ->bodyAttribute('flex mt-2 mx-3'),

            Column::make(t('created_at'), 'created_at_formatted', 'created_at')
                ->sortable(),
        ];
    }

    public function filters(): array
    {
        return [
            // Type filter
            Filter::select('type')
                ->dataSource(collect(WhatsAppTemplateRelationType::getRelationtype())
                    ->map(fn ($value, $key) => ['value' => $key, 'label' => ucfirst($value)])
                    ->values()
                    ->toArray())
                ->optionValue('value')
                ->optionLabel('label'),

            // Assigned User filter - cached
            Filter::select('assigned_id')
                ->dataSource(
                    TenantCache::remember(self::CACHE_KEY_USERS, self::CACHE_DURATION, function () {
                        return User::query()
                            ->select(['id', 'firstname', 'lastname'])
                            ->where('tenant_id', $this->tenant_id)
                            ->get()
                            ->map(fn ($user) => [
                                'id' => $user->id,
                                'name' => $user->firstname.' '.$user->lastname,
                            ]);
                    }, ['contact-filters', 'users'])
                )
                ->optionValue('id')
                ->optionLabel('name'),

            // Status filter - cached
            Filter::select('status_id')
                ->dataSource(
                    TenantCache::remember(self::CACHE_KEY_STATUSES.'_'.$this->tenant_subdomain, self::CACHE_DURATION, function () {
                        return Status::where('tenant_id', $this->tenant_id)
                            ->select(['id', 'name'])
                            ->get()
                            ->toArray();
                    }, ['contact-filters', 'statuses'])
                )
                ->optionValue('id')
                ->optionLabel('name'),

            Filter::select('source_id')
                ->dataSource(
                    TenantCache::remember(self::CACHE_KEY_SOURCES.'_'.$this->tenant_subdomain, self::CACHE_DURATION, function () {
                        return Source::where('tenant_id', $this->tenant_id)
                            ->select(['id', 'name'])
                            ->get()
                            ->toArray();
                    }, ['contact-filters', 'sources'])
                )
                ->optionValue('id')
                ->optionLabel('name'),
        ];
    }

    #[On('bulkDelete.{tableName}')]
    public function bulkDelete(): void
    {
        $selectedIds = $this->checkboxValues;
        if (! empty($selectedIds) && count($selectedIds) !== 0) {
            $this->dispatch('confirmDelete', $selectedIds);
            $this->checkboxValues = [];
        } else {
            $this->notify(['type' => 'danger', 'message' => t('no_contact_selected')]);
        }
    }

    public function onUpdatedToggleable(string $id, string $field, string $value): void
    {
        if (checkPermission('tenant.contact.edit')) {
            $this->dispatch('refreshComponent');

            Contact::fromTenant($this->tenant_subdomain)->where('id', $id)->update(['is_enabled' => $value === '1' ? 1 : 0]);

            $this->notify([
                'message' => $value === '1' ? t('contact_enable_successfully') : t('contact_disabled_successfully'),
                'type' => 'success',
            ]);
        } else {
            $this->notify([
                'message' => t('access_denied_note'),
                'type' => 'warning',
            ]);
        }
    }

    /*
    public function actionRules($row): array
    {
       return [
            // Hide button edit for ID 1
            Rule::button('edit')
                ->when(fn($row) => $row->id === 1)
                ->hide(),
        ];
    }
    */
}
