<?php

namespace App\Livewire\Tenant\Settings\System;

use App\Facades\TenantCache;
use App\Rules\PurifiedInput;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithFileUploads;

class GeneralSettings extends Component
{
    use WithFileUploads;

    public ?string $site_name = '';

    public ?string $site_description = '';

    public ?string $timezone = '';

    public ?string $date_format = '';

    public ?string $time_format = '';

    public ?string $active_language = '';

    public ?array $timezone_list = [];

    public array $date_formats = [
        'Y-m-d' => 'Y-m-d',
        'd/m/Y' => 'd/m/Y',
        'm/d/Y' => 'm/d/Y',
        'd.m.Y' => 'd.m.Y',
        'd-m-Y' => 'd-m-Y',
        'm-d-Y' => 'm-d-Y',
        'm.d.Y' => 'm.d.Y',
    ];

    public $company_light_logo;

    public $company_dark_logo;

    public $favicon;

    public $temp_company_light_logo;

    public $temp_company_dark_logo;

    public $temp_favicon;

    public $current_company_light_logo;

    public $current_company_dark_logo;

    public $current_favicon;

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    protected function rules()
    {
        return [
            'site_name' => ['nullable', 'string', new PurifiedInput(t('sql_injection_error'))],
            'site_description' => ['nullable', 'string', new PurifiedInput(t('sql_injection_error'))],
            'timezone' => 'nullable|string|timezone',
            'date_format' => 'nullable|string',
            'active_language' => 'nullable|string',
            'time_format' => 'nullable|string',
            'company_light_logo' => 'nullable|image',
            'company_dark_logo' => 'nullable|image',
            'favicon' => 'nullable|image',

        ];
    }

    public function mount()
    {
        if (! checkPermission('tenant.system_settings.view')) {
            $this->notify(['type' => 'danger', 'message' => t('access_denied_note')], true);

            return redirect()->to(tenant_route('tenant.dashboard'));
        }
        $this->timezone_list = timezone_identifiers_list();

        $system_settings = tenant_settings_by_group('system');

        $this->site_name = $system_settings['site_name'] ?? '';
        $this->site_description = $system_settings['site_description'] ?? '';
        $this->timezone = $system_settings['timezone'] ?? '';
        $this->date_format = $system_settings['date_format'] ?? '';
        $this->time_format = $system_settings['time_format'] ?? '';
        $this->active_language = $system_settings['active_language'] ?? '';
        $this->current_company_light_logo = $system_settings['company_light_logo'] ?? '';
        $this->current_company_dark_logo = $system_settings['company_dark_logo'] ?? '';
        $this->current_favicon = $system_settings['favicon'] ?? '';
    }

    public function removeImage($type)
    {
        if (checkPermission('tenant.system_settings.edit')) {
            if ($type === 'company_light_logo') {
                if ($this->current_company_light_logo && Storage::disk('public')->exists($this->current_company_light_logo)) {
                    Storage::disk('public')->delete($this->current_company_light_logo);
                }
                save_tenant_setting('system', 'company_light_logo', '');
                $this->current_company_light_logo = '';
            } elseif ($type === 'favicon') {
                if ($this->current_favicon && Storage::disk('public')->exists($this->current_favicon)) {
                    Storage::disk('public')->delete($this->current_favicon);
                }
                save_tenant_setting('system', 'favicon', '');
                $this->current_favicon = '';
            } elseif ($type === 'company_dark_logo') {
                if ($this->current_company_dark_logo && Storage::disk('public')->exists($this->current_company_dark_logo)) {
                    Storage::disk('public')->delete($this->current_company_dark_logo);
                }
                save_tenant_setting('system', 'company_dark_logo', '');
                $this->current_company_dark_logo = '';
            }

            $this->notify([
                'type' => 'success',
                'message' => t('image_removed_successfully'),
            ], true);

            return redirect()->to(tenant_route('tenant.settings.general'));
        } else {
            $this->notify([
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(tenant_route('tenant.settings.general'));
        }
    }

    public function save()
    {
        if (checkPermission('tenant.system_settings.edit')) {
            $this->validate();

            $tenant_id = tenant_id();
            $originalSettings = tenant_settings_by_group('system');
            $language = $originalSettings['active_language'] ?? 'en';
            TenantCache::forget("translations.{$tenant_id}_tenant_{$language}");

            $imageChanged = false;

            // Handle logo upload
            if ($this->temp_company_light_logo) {
                if ($this->current_company_light_logo && Storage::disk('public')->exists($this->current_company_light_logo)) {
                    Storage::disk('public')->delete($this->current_company_light_logo);
                }
                $company_logo_path = $this->temp_company_light_logo->store('tenant/'.$tenant_id.'/company_logos', 'public');
                save_tenant_setting('system', 'company_light_logo', $company_logo_path);
                $this->current_company_light_logo = $company_logo_path;
                $this->temp_company_light_logo = null;
                $imageChanged = true;
            }

            // Handle favicon upload
            if ($this->temp_favicon) {
                if ($this->current_favicon && Storage::disk('public')->exists($this->current_favicon)) {
                    Storage::disk('public')->delete($this->current_favicon);
                }
                $favicon_path = $this->temp_favicon->store('tenant/'.$tenant_id.'/favicons', 'public');
                save_tenant_setting('system', 'favicon', $favicon_path);
                $this->current_favicon = $favicon_path;
                $this->temp_favicon = null;
                $imageChanged = true;
            }
            if ($this->temp_company_dark_logo) {
                if ($this->current_company_dark_logo && Storage::disk('public')->exists($this->current_company_dark_logo)) {
                    Storage::disk('public')->delete($this->current_company_dark_logo);
                }
                $dark_logo_path = $this->temp_company_dark_logo->store('tenant/'.$tenant_id.'/company_logos', 'public');
                save_tenant_setting('system', 'company_dark_logo', $dark_logo_path);
                $this->current_company_dark_logo = $dark_logo_path;
                $this->temp_company_dark_logo = null;
                $imageChanged = true;
            }

            $newSettings = [
                'site_name' => $this->site_name,
                'site_description' => $this->site_description,
                'timezone' => $this->timezone,
                'date_format' => $this->date_format,
                'time_format' => $this->time_format,
                'active_language' => $this->active_language,
            ];

            Session::put('locale', $this->active_language);
            App::setLocale($this->active_language);

            // Compare and filter only modified or missing settings
            $modifiedSettings = array_filter($newSettings, function ($value, $key) use ($originalSettings) {
                return ! array_key_exists($key, $originalSettings) || $originalSettings[$key] !== $value;
            }, ARRAY_FILTER_USE_BOTH);

            if (! empty($modifiedSettings)) {
                foreach ($modifiedSettings as $key => $value) {
                    save_tenant_setting('system', $key, $value);
                }
            }

            // Always notify and dispatch after any change (image or text)
            if ($imageChanged || ! empty($modifiedSettings)) {
                $this->dispatch('imageUploaded', [
                    'company_light_logo' => $this->current_company_light_logo ? asset('storage/'.$this->current_company_light_logo) : null,
                    'favicon' => $this->current_favicon ? asset('storage/'.$this->current_favicon) : null,
                ]);

                $this->notify([
                    'type' => 'success',
                    'message' => t('setting_save_successfully'),
                ], true);

                return redirect()->to(tenant_route('tenant.settings.general'));
            }
        }
    }

    public function render()
    {
        return view('livewire.tenant.settings.system.general-settings');
    }
}
