<?php

namespace App\Jobs;

use App\Models\Subscription;
use App\Services\ErrorHandler;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessSubscriptionRenewal implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The subscription instance.
     *
     * @var \App\Models\Subscription
     */
    protected $subscription;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The maximum number of unhandled exceptions to allow before failing.
     *
     * @var int
     */
    public $maxExceptions = 1;

    /**
     * Indicate if the job should be marked as failed on timeout.
     *
     * @var bool
     */
    public $failOnTimeout = true;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 60;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Subscription $subscription)
    {
        $this->subscription = $subscription;
        $this->onQueue('subscriptions');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {

            // Check if the subscription is still active
            if (! $this->subscription->isActive()) {
                return;
            }

            // Create renewal invoice if needed
            $invoice = $this->subscription->checkAndCreateRenewInvoice();

            if ($invoice) {

                // If the customer has auto-billing enabled, dispatch a job to process payment
                if ($this->subscription->is_recurring && $this->subscription->customer->getAutoBillingData()) {
                    ProcessInvoicePayment::dispatch($invoice)->onQueue('payments');

                }
            }

        } catch (\Exception $e) {
            $errorDetails = ErrorHandler::handleSubscriptionError($e, 'RENEWAL_JOB');

            app_log('Failed to process subscription renewal', 'error', $e, [
                'subscription_id' => $this->subscription->id,
                'error_details' => $errorDetails,
            ]);

            // Rethrow the exception to retry the job
            throw $e;
        }
    }
}
